import Optimizer
#from matplotlib import pyplot as plt
import math
import copy

# abstract class used to specify interface
class Evaluator:
    # ABSTRACT: intializes the characterizer with its required data
    # return type: self
    def __init__(self,measurementDatabase,verbose=False):
        raise Exception("Optimizer initialization is not defined")
        # returns type: None

    # ABSTRACT: characterizes a state's phased array performance
    # note that a LARGER, more POSITIVE, number is a BETTER performance
    # returns type: float
    def characterize(self, state):
        raise Exception("Characterizer characterize is not defined")

# Maximizes the main lobe over the steering range
# FOM is the average main beam power over the steering range
class MaximizeMainLobe(Evaluator):
    # Intializes the characterizer with its required data
    # return type: self
    def __init__(self,measurementDatabase,theta_b_vals, phi_b_vals,theta_m_vals,phi_m_vals,in_db = False, verbose=False):
        self.measDB = measurementDatabase
        self.theta_b_vals = theta_b_vals
        self.phi_b_vals = phi_b_vals
        self.theta_m_vals = theta_m_vals
        self.phi_m_vals = phi_m_vals
        self.in_db = in_db

    # Characterizes a state's phased array performance
    # note that a LARGER, more POSITIVE, number is a BETTER performance
    # returns eta, type: float
    def characterize(self, state,baseline_state):
        eta = 0
        for phi_b in self.phi_b_vals:
            phi_m_idx = self.phi_m_vals.index(phi_b)
            for theta_b in self.theta_b_vals:
                theta_m_idx = self.theta_m_vals.index(theta_b)
                # deep copy so that we do not modify database
                beam_pattern = copy.deepcopy(self.measDB.getPatternData(state,theta_b,phi_b))
                # deep copy so that we do not modify database
                baseline_pattern = copy.deepcopy(self.measDB.getPatternData(baseline_state,theta_b,phi_b))

                pattern_theta_slice = beam_pattern[phi_m_idx]
                baseline_pattern_theta_slice = baseline_pattern[phi_m_idx]

                normalized_pattern = []
                for i in range(len(self.theta_m_vals)):
                    beam_val = pattern_theta_slice[i]
                    baseline_val = baseline_pattern_theta_slice[i]
                    if not self.in_db:
                        beam_val = 10**(beam_val/10.0)
                        baseline_val = 10**(baseline_val/10.0)
                    normalized_pattern += [beam_val - baseline_val]

                main_lobe_value = normalized_pattern[theta_m_idx]
                eta += main_lobe_value
        eta = eta / float(len(self.phi_b_vals)*len(self.theta_b_vals))
        self.measDB.saveCharacterizationData(state,eta)
        return eta


# Maximizes the peak side lobe relative to the maximum main lobe
# FOM is the average negative SLL over the steering range
class MinimizePeakSideLobeRelative(Evaluator):
    # Intializes the characterizer with its required data
    # return type: self
    def __init__(self,measurementDatabase,theta_b_vals, phi_b_vals,theta_m_vals,phi_m_vals,beam_width,in_db = False, verbose=False):
        self.measDB = measurementDatabase
        self.theta_b_vals = theta_b_vals
        self.phi_b_vals = phi_b_vals
        self.theta_m_vals = theta_m_vals
        self.phi_m_vals = phi_m_vals
        self.in_db = in_db
        receiver_spacing = theta_m_vals[1]-theta_m_vals[0]
        self.half_beam_width_index = math.floor((beam_width/2)/receiver_spacing)

    # Characterizes a state's phased array performance
    # note that a LARGER, more POSITIVE, number is a BETTER performance
    # returns eta, type: float
    # THIS CAN BE OPTIMIZED LIKE I DID IN MATLAB CODE
    def characterize(self, state, baseline_state):
        eta = 0
        for phi_b in self.phi_b_vals:
            phi_m_idx = self.phi_m_vals.index(phi_b)
            for theta_b in self.theta_b_vals:
                theta_m_idx = self.theta_m_vals.index(theta_b)
                # deep copy so that we do not modify database
                beam_pattern = copy.deepcopy(self.measDB.getPatternData(state,theta_b,phi_b))
                # deep copy so that we do not modify database
                baseline_pattern = copy.deepcopy(self.measDB.getPatternData(baseline_state,theta_b,phi_b))

                pattern_theta_slice = beam_pattern[phi_m_idx]
                baseline_pattern_theta_slice = baseline_pattern[phi_m_idx]

                normalized_pattern = []
                for i in range(len(self.theta_m_vals)):
                    beam_val = pattern_theta_slice[i]
                    baseline_val = baseline_pattern_theta_slice[i]
                    if not self.in_db:
                        beam_val = 10**(beam_val/10.0)
                        baseline_val = 10**(baseline_val/10.0)
                    normalized_pattern += [beam_val - baseline_val]

                main_lobe_value = normalized_pattern[theta_m_idx]

                # get minium and maximum beam width
                min_rec_idx = int(max(0,theta_m_idx-self.half_beam_width_index))
                max_rec_idx = int(min(len(self.theta_m_vals),theta_m_idx+self.half_beam_width_index+1))

                # nulify main lobe
                nul_val = -100
                if not self.in_db:
                        nul_val = 0
                normalized_pattern[min_rec_idx:max_rec_idx] = [nul_val]*(max_rec_idx-min_rec_idx)
                # get peak side lobe
                peak_side_lobe = max(normalized_pattern)

                peak_side_lobe_relative = peak_side_lobe-main_lobe_value
                # we want to minimize, so negate
                eta += -1*peak_side_lobe_relative
        eta = eta / float(len(self.phi_b_vals)*len(self.theta_b_vals))
        self.measDB.saveCharacterizationData(state,eta)
        return eta

# Maximizes the fov as defined as the angular coverage for which the main lobe is larger than the peak side lobe
# FOM is the average FOV over the different phi cuts
class MaximizeFieldOfView(Evaluator):
    # Intializes the characterizer with its required data
    # return type: self
    def __init__(self,measurementDatabase,theta_b_vals, phi_b_vals,theta_m_vals,phi_m_vals,beam_width,in_db = False,verbose=False):
        self.measDB = measurementDatabase
        self.theta_b_vals = theta_b_vals
        self.phi_b_vals = phi_b_vals
        self.theta_m_vals = theta_m_vals
        self.phi_m_vals = phi_m_vals
        self.in_db = in_db
        receiver_spacing = theta_m_vals[1]-theta_m_vals[0]
        self.half_beam_width_index = math.floor((beam_width/2)/receiver_spacing)

    # Characterizes a state's phased array performance
    # note that a LARGER, more POSITIVE, number is a BETTER performance
    # returns eta, type: float
    def characterize(self, state, baseline_state):
        eta = 0
        for phi_b in self.phi_b_vals:
            psls_rel = []
            phi_m_idx = self.phi_m_vals.index(phi_b)
            for theta_b in self.theta_b_vals:
                theta_m_idx = self.theta_m_vals.index(theta_b)
                # deep copy so that we do not modify database
                beam_pattern = copy.deepcopy(self.measDB.getPatternData(state,theta_b,phi_b))

                pattern_theta_slice = beam_pattern[phi_m_idx]

                normalized_pattern = []
                for i in range(len(self.theta_m_vals)):
                    beam_val = pattern_theta_slice[i]
                    if not self.in_db:
                        beam_val = 10**(beam_val/10.0)
                    normalized_pattern += [beam_val]

                main_lobe_value = normalized_pattern[theta_m_idx]

                # get minium and maximum beam width
                min_rec_idx = int(max(0,theta_m_idx-self.half_beam_width_index))
                max_rec_idx = int(min(len(self.theta_m_vals),theta_m_idx+self.half_beam_width_index+1))

                # nulify main lobe
                nul_val = -100
                if not self.in_db:
                        nul_val = 0
                normalized_pattern[min_rec_idx:max_rec_idx] = [nul_val]*(max_rec_idx-min_rec_idx)

                # get peak side lobe
                peak_side_lobe = max(normalized_pattern)

                peak_side_lobe_relative = peak_side_lobe-main_lobe_value
                psls_rel += [peak_side_lobe_relative]
            center_idx = int(math.floor(len(self.theta_b_vals)/2.0 + 0.5)-1);
            # assume that psl is larger outside of measurement zone
            psl_greater_than_ml = True
            # first index after cross into FOV
            first_cross_p2_idx = center_idx
            # moving left towards broadside
            for theta_b_idx in range(center_idx+1):
                pslr = psls_rel[theta_b_idx]
                # check for positive crossing, if so then this could be start of FOV
                if (pslr < 0) and psl_greater_than_ml:
                    psl_greater_than_ml = False
                    first_cross_p2_idx = theta_b_idx
                # check for negative crossing, if so then this cannot be in FOV
                if (pslr >= 0) and not psl_greater_than_ml:
                    psl_greater_than_ml = True
                    first_cross_p2_idx = center_idx

            # assume that psl is larger outside of measurement zone
            psl_greater_than_ml = True
            # last index before cross out of FOV
            last_cross_p1_idx = center_idx
            # moving right towards broadside
            for theta_b_idx in range(len(self.theta_b_vals)-1, center_idx-1, -1):
                pslr = psls_rel[theta_b_idx]
                # check for positive crossing, if so then this could be end of FOV
                if (pslr < 0) and psl_greater_than_ml:
                    psl_greater_than_ml = False
                    last_cross_p1_idx = theta_b_idx
                # check for negative crossing, if so then this cannot be in FOV
                if (pslr >= 0) and not psl_greater_than_ml:
                    psl_greater_than_ml = True
                    last_cross_p1_idx = center_idx

            # crossover happens before first index
            first_cross_p1_idx = first_cross_p2_idx - 1
            # crossover happens after last index
            last_cross_p2_idx = last_cross_p1_idx +1

            if first_cross_p1_idx >= 0:
                ang1 = self.theta_b_vals[first_cross_p1_idx]
                ang2 = self.theta_b_vals[first_cross_p2_idx]
                pslr1 = psls_rel[first_cross_p1_idx]
                pslr2 = psls_rel[first_cross_p2_idx]
                d_ang  = ang2 - ang1
                if (pslr1 - pslr2) == 0:
                    first_cross_ang = ang2
                else:
                    first_cross_ang = d_ang * pslr1 / (pslr1 - pslr2) + ang1;
            else:
                first_cross_ang = self.theta_b_vals[0]

            if last_cross_p2_idx < len(self.theta_b_vals):
                ang1 = self.theta_b_vals[last_cross_p1_idx]
                ang2 = self.theta_b_vals[last_cross_p2_idx]
                pslr1 = psls_rel[last_cross_p1_idx]
                pslr2 = psls_rel[last_cross_p2_idx]
                d_ang  = ang2 - ang1
                if (pslr1 - pslr2) == 0:
                    last_cross_ang = ang1
                else:
                    last_cross_ang = d_ang * pslr1 / (pslr1 - pslr2) + ang1;
            else:
                last_cross_ang = self.theta_b_vals[len(self.theta_b_vals)-1]

            field_of_view = last_cross_ang - first_cross_ang
            eta += field_of_view
        eta = eta / float(len(self.phi_b_vals))
        self.measDB.saveCharacterizationData(state,eta)
        return eta
